/*
Copyright 2008-2009 Elöd Egyed-Zsigmond, Cyril Laitang
Copyright 2009-2011 Samuel Gesche

This file is part of IPRI News Analyzer.

IPRI News Analyzer is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

IPRI News Analyzer is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with IPRI News Analyzer.  If not, see <http://www.gnu.org/licenses/>.
*/

package data.base.connectors;

import data.base.Database;
import data.base.NoBaseException;
import data.structures.rss.SujetBrut;
import data.structures.rss.SujetTraite;
import data.structures.rss.Tag;
import data.structures.rss.RSSFeedInfo;
import data.structures.rss.RSSFeedInfoExtended;
import data.structures.tagging.LemmaItem;
import data.structures.rss.BaseRSSItem;
import data.structures.rss.RSSItem;
import data.structures.rss.RSSItemCount;
import data.structures.analyse.SousCorpus;

import proc.text.Codecs;
import proc.text.Out;

import java.io.File;
import java.io.IOException;

import java.sql.ResultSet;
import java.sql.SQLException;

import java.util.Set;
import java.util.Vector;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.GregorianCalendar;

public class RSSFeedDatabase {

    public final static int TAGS_CATEGORIE = 100;
    public final static int TAGS_PAYS = 200;
    public final static int TAGS_COUVERTURE = 300;
    public final static int TAGS_INFORMATION = 400;
    public final static int TAGS_EXTERNE = 900;

    private Database theDB;
    /*private static List ignoreLemma;*/
    //Optimisation : inutile de rechercher la date tout le temps si elle 
    //ne change qu'en cas de nouvelle entrée.
    private static Date[] lastFeedDates;
    private static Date lastNewsDates;

    public RSSFeedDatabase(Database db) throws NoBaseException {
        theDB = db;
        // populate de la liste de lemmes à eliminer
        /*ignoreLemma = new ArrayList();
        try {
            getIgnoreLemma();
        } catch (SQLException ex) {
            ex.printStackTrace();
        }*/
        if(lastFeedDates == null){
            Vector<RSSFeedInfo> feedList = GetRSSFeedList();
            int max = 0;
            for (int i = 0; i < feedList.size(); i++) {
                int id = Integer.parseInt(feedList.elementAt(i).ID);
                if (id > max) {
                    max = id;
                }
            }
            lastFeedDates = new Date[max+1];
        }
    }

    public Tag[] getTags(int type) throws NoBaseException {
        String query = "SELECT id, tag FROM rss_tags WHERE id BETWEEN "+type+" AND "+(type+99)+";";
        ResultSet tags = theDB.executeSelection(query);

        Vector<Tag> res = new Vector<Tag>();
        if (tags != null) {
            try {
                while (tags.next()) { // process results one row at a time

                    int id = tags.getInt("id");
                    String tag = Codecs.desEscapeHTML(tags.getString("tag"));
                    res.addElement(new Tag(id, tag));
                }
                tags.close();
            } catch (Exception e) {
                return new Tag[]{};
            }
        }
        Tag[] res2 = new Tag[res.size()];
        res.toArray(res2);
        return res2;
    }

    public Tag[] getTagsOfFeed(int feedId, int type) throws NoBaseException {
        String query = "SELECT t.id, t.tag FROM rss_tags t, rss_tagflux f " +
                "WHERE f.flux="+feedId+" AND f.tag=t.id " +
                "AND t.id BETWEEN "+type+" AND "+(type+99)+";";
        ResultSet tags = theDB.executeSelection(query);

        Vector<Tag> res = new Vector<Tag>();
        if (tags != null) {
            try {
                while (tags.next()) { // process results one row at a time

                    int id = tags.getInt("id");
                    String tag = Codecs.desEscapeHTML(tags.getString("tag"));
                    res.addElement(new Tag(id, tag));
                }
                tags.close();
            } catch (Exception e) {
                return new Tag[]{};
            }
        }
        Tag[] res2 = new Tag[res.size()];
        res.toArray(res2);
        return res2;
    }
    
    private Map<Integer, String> feedNames = new HashMap<Integer, String>();
    public String getFeedName(int feedId) throws NoBaseException {
        String res = feedNames.get(new Integer(feedId));
        if (res == null) {
            String query = "SELECT nom FROM rss_flux WHERE id=" + feedId + ";";
            ResultSet feedName = theDB.executeSelection(query);
            if (feedName != null) {
                try {
                    while (feedName.next()) {
                        String nom = Codecs.desEscapeHTML(feedName.getString("nom"));
                        res = nom;
                    }
                    feedName.close();
                } catch (Exception e) {
                    res = "";
                }
                feedNames.put(new Integer(feedId), res);
            }
        }
        return res;
    }

    private Map<String, Integer> feedIds = new HashMap<String, Integer>();
    public int findFeedId(String feedName) throws NoBaseException {
        Integer res = feedIds.get(feedName);
        if (res == null) {
            String query = "SELECT id FROM rss_flux " +
                    "WHERE nom=\"" + feedName + "\" LIMIT 1;";
            ResultSet feedId = theDB.executeSelection(query);
            if (feedId != null) {
                try {
                    while (feedId.next()) {
                        int id = feedId.getInt("id");
                        res = new Integer(id);
                    }
                    feedId.close();
                } catch (Exception e) {
                    
                }
                if(res == null){
                    res = new Integer(-1);
                }
                feedIds.put(feedName, res);
            }
        }
        return res.intValue();
    }

    private Map<Integer, String> feedUrls = new HashMap<Integer, String>();
    public String getFeedURL(int feedId) throws NoBaseException {
        String res = feedUrls.get(new Integer(feedId));
        if (res == null) {
            String query = "SELECT url FROM rss_flux WHERE id=" + feedId + ";";
            ResultSet feedName = theDB.executeSelection(query);
            if (feedName != null) {
                try {
                    while (feedName.next()) {
                        String url = Codecs.desEscapeHTML(feedName.getString("url"));
                        res = url;
                    }
                    feedName.close();
                } catch (Exception e) {
                    res = "";
                }
                feedUrls.put(new Integer(feedId), res);
            }
        }
        return res;
    }

    public Vector<RSSFeedInfo> GetRSSFeedList() throws NoBaseException {

        Vector<RSSFeedInfo> laListe = new Vector<RSSFeedInfo>();

        ResultSet feeds = theDB.executeSelection("SELECT * from rss_flux;");
        if (feeds != null) {
            try {
                while (feeds.next()) { // process results one row at a time
                    String id = Codecs.desEscapeHTML(feeds.getString("id"));
                    String url = Codecs.desEscapeHTML(feeds.getString("url"));

                    RSSFeedInfo inf = new RSSFeedInfo();
                    inf.ID = id;
                    inf.URL = url;
                    laListe.add(inf);
                }
                feeds.close();
            } catch (Exception e) {
                laListe = null;
            }
        }
        return laListe;
    }

    public Vector<RSSFeedInfoExtended> GetExtendedRSSFeedList() throws NoBaseException {

        Vector<RSSFeedInfoExtended> laListe = new Vector<RSSFeedInfoExtended>();

        ResultSet feeds = theDB.executeSelection("SELECT id, url, nom from rss_flux;");
        if (feeds != null) {
            try {
                while (feeds.next()) { // process results one row at a time

                    String id = Codecs.desEscapeHTML(feeds.getString(1));
                    String url = Codecs.desEscapeHTML(feeds.getString(2));
                    String name = Codecs.desEscapeHTML(feeds.getString(3));

                    RSSFeedInfoExtended inf = new RSSFeedInfoExtended();
                    inf.ID = id;
                    inf.URL = url;
                    inf.name = name;
                    laListe.add(inf);
                }
                feeds.close();
            } catch (Exception e) {
                return null;
            }
        }
        return laListe;
    }

    public Vector<RSSFeedInfoExtended> getExtendedRSSFeedList(Tag[] tagsAcceptes) throws NoBaseException {
        Vector<Tag> listeCategories = new Vector<Tag>();
        Vector<Tag> listePays = new Vector<Tag>();
        Vector<Tag> listeCouvertures = new Vector<Tag>();
        Vector<Tag> listeInformations = new Vector<Tag>();
        Vector<Tag> listeExterne = new Vector<Tag>();

        for(int i=0; i<tagsAcceptes.length; i++){
            if(tagsAcceptes[i].getId()>=TAGS_CATEGORIE && tagsAcceptes[i].getId()<TAGS_PAYS){
                listeCategories.addElement(tagsAcceptes[i]);
            } else if(tagsAcceptes[i].getId()>=TAGS_PAYS && tagsAcceptes[i].getId()<TAGS_COUVERTURE){
                listePays.addElement(tagsAcceptes[i]);
            } else if(tagsAcceptes[i].getId()>=TAGS_COUVERTURE && tagsAcceptes[i].getId()<TAGS_INFORMATION){
                listeCouvertures.addElement(tagsAcceptes[i]);
            } else if(tagsAcceptes[i].getId()>=TAGS_INFORMATION && tagsAcceptes[i].getId()<TAGS_EXTERNE){
                listeInformations.addElement(tagsAcceptes[i]);
            } else {
                listeExterne.addElement(tagsAcceptes[i]);
            }
        }
        
        String s = "SELECT id, url, nom FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeCategories.size(); i++){
            s += ""+listeCategories.elementAt(i).getId();
            if(i < listeCategories.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listePays.size(); i++){
            s += ""+listePays.elementAt(i).getId();
            if(i < listePays.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeCouvertures.size(); i++){
            s += ""+listeCouvertures.elementAt(i).getId();
            if(i < listeCouvertures.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeInformations.size(); i++){
            s += ""+listeInformations.elementAt(i).getId();
            if(i < listeInformations.size()-1){
                s += " OR t.tag=";
            }
        }
        /*s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeExterne.size(); i++){
            s += ""+listeExterne.elementAt(i).getId();
            if(i < listeExterne.size()-1){
                s += " OR t.tag=";
            }
        }*/
        s += /*")*/"))));";

        Vector<RSSFeedInfoExtended> laListe = new Vector<RSSFeedInfoExtended>();

        ResultSet feeds = theDB.executeSelection(s);
        if (feeds != null) {
            try {
                while (feeds.next()) { // process results one row at a time

                    String id = Codecs.desEscapeHTML(feeds.getString(1));
                    String url = Codecs.desEscapeHTML(feeds.getString(2));
                    String name = Codecs.desEscapeHTML(feeds.getString(3));

                    RSSFeedInfoExtended inf = new RSSFeedInfoExtended();
                    inf.ID = id;
                    inf.URL = url;
                    inf.name = name;
                    laListe.add(inf);
                }
                feeds.close();
            } catch (Exception e) {
                return null;
            }
        }
        return laListe;
    }

    public Date GetFeedLastEntryDate(int feedID) throws NoBaseException {
        if (lastFeedDates[feedID] == null) {
            Date theDate = new Date(data.base.Config.getDateDepartCorpus());
            String query = "SELECT MAX(datePubli) FROM rss_articles WHERE flux = " + feedID;
            String queryCount = "SELECT COUNT(datePubli) FROM rss_articles  WHERE flux = " +  feedID;
            ResultSet dateCount = theDB.executeSelection(queryCount);

            boolean areDates = false;
            try {
                areDates = dateCount.next();
                areDates = (dateCount.getInt(1) > 0);
                dateCount.close();
            } catch (Exception e) {
                return theDate;
            }
            ResultSet dateRes = theDB.executeSelection(query);
            if (areDates) {
                try {
                    while (dateRes.next()) { // process results one row at a time

                        theDate = dateRes.getTimestamp(1);
                    }
                    dateRes.close();
                } catch (Exception e) {
                    return theDate;
                }
            }
            lastFeedDates[feedID] = theDate;
            return theDate;
        } else {
            return lastFeedDates[feedID];
        }
    }

    public RSSItem[] listeArticlesRSS(Date debut, Date fin, SousCorpus tagsAcceptes) throws NoBaseException {
        Vector<Tag> listeCategories = new Vector<Tag>();
        Vector<Tag> listePays = new Vector<Tag>();
        Vector<Tag> listeCouvertures = new Vector<Tag>();
        Vector<Tag> listeInformations = new Vector<Tag>();

        for(int i=0; i<tagsAcceptes.listeTagsAcceptes().length; i++){
            if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_CATEGORIE && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_PAYS){
                listeCategories.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_PAYS && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_COUVERTURE){
                listePays.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_COUVERTURE && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_INFORMATION){
                listeCouvertures.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_INFORMATION && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_EXTERNE){
                listeInformations.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            }
        }

        String s = String.format("SELECT * FROM rss_articles WHERE datePubli >= '%tY-%tm-%td 00:00:00' " +
                "AND datePubli < '%tY-%tm-%td 00:00:00' AND flux IN (" +
                "SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND ", debut, debut, debut, fin, fin, fin);
        s += "(t.tag=";
        for(int i=0; i<listeCategories.size(); i++){
            s += ""+listeCategories.elementAt(i).getId();
            if(i < listeCategories.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listePays.size(); i++){
            s += ""+listePays.elementAt(i).getId();
            if(i < listePays.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeCouvertures.size(); i++){
            s += ""+listeCouvertures.elementAt(i).getId();
            if(i < listeCouvertures.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeInformations.size(); i++){
            s += ""+listeInformations.elementAt(i).getId();
            if(i < listeInformations.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ")))));";
        Vector<RSSItem> res = new Vector<RSSItem>();
        ResultSet articles = theDB.executeSelection(s);
        try {
            while (articles.next()) { // process results one row at a time
                RSSItem article = new RSSItem();
                article.setId(new Integer(articles.getInt("id")));
                article.setIdFluxRSS(new Integer(articles.getInt("flux")));
                article.setPubDate(articles.getTimestamp("datePubli"));
                article.setPubTime(articles.getTime("datePubli"));
                article.setTitle(Codecs.desEscapeHTML(articles.getString("titre")));
                article.setDescription(Codecs.desEscapeHTML(articles.getString("description")));
                article.setLink(Codecs.desEscapeURL(articles.getString("url")));
                article.setTitleLemmes(Codecs.desEscapeHTML(articles.getString("titreLemma")));
                article.setDescLemmes(Codecs.desEscapeHTML(articles.getString("descLemma")));
                res.addElement(article);
            }
            articles.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
        RSSItem[] res2 = new RSSItem[res.size()];
        res.toArray(res2);
        for(int i=0; i<res2.length; i++){
            res2[i].setNomFluxRSS(getFeedName(res2[i].getIdFluxRSS()));
        }
        return res2;
    }

    public void insereSujet(String titre, BaseRSSItem[] news, Date datePubli) throws NoBaseException {
        // rechercher les articles correspondants aux URLs
        /*int[] articles = new int[urls.length];
        for(int i=0; i<urls.length; i++){
            articles[i] = -1;
            String query = "SELECT id FROM rss_articles WHERE url='"+urls[i]+"';";
            ResultSet url = theDB.executeSelection(query);
            try {
                if (url.next()) {
                    articles[i] = url.getInt("id");
                }
                url.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }*/
        // enregistrer le nouveau sujet
        String query = "SELECT MAX(id) FROM rss_sujets;";
        ResultSet rsIdMax = theDB.executeSelection(query);
        int idMax = 0;
        try {
            if (rsIdMax.next()) {
                idMax = rsIdMax.getInt(1);
            }
            rsIdMax.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
        idMax++;
        String query2 = String.format("INSERT INTO rss_sujets (id, titre, datePubli) " +
                "VALUES (%d, '%s', '%tY-%tm-%td %tH:%tM:%tS');",
                idMax,
                Codecs.escapeHTML(titre),
                datePubli, datePubli, datePubli, datePubli, datePubli, datePubli);
        theDB.executeInsertOrUpdate(query2);
        lastNewsDates = null;
        // enregistrer les URLs
        for(int i=0; i<news.length; i++){
            if(!(news[i].getLink().trim().equals(""))){
                String queryArticle = "INSERT INTO rss_articlessujets " +
                        "(sujet, titre, description, url) VALUES(" +
                        idMax + ", '" +
                        Codecs.escapeHTML(news[i].getTitle())+ "', '" +
                        Codecs.escapeHTML(news[i].getDescription())+ "', '" +
                        Codecs.echappeSQL(Codecs.desEscapeURL(news[i].getLink())) + "');";
                theDB.executeInsertOrUpdate(queryArticle);
            } else {
                System.err.println("Erreur : récupération d'url de sujets inefficace");
            }
        }
    }

    public Date getNewsLastEntryDate() throws NoBaseException {
        if (lastNewsDates == null) {
            Date theDate = new Date(data.base.Config.getDateDepartCorpus());
            String query = "SELECT MAX(datePubli) FROM rss_sujets;";
            String queryCount = "SELECT COUNT(datePubli) FROM rss_sujets;";
            ResultSet dateCount = theDB.executeSelection(queryCount);

            boolean areDates = false;
            try {
                areDates = dateCount.next();
                areDates = (dateCount.getInt(1) > 0);
                dateCount.close();
            } catch (Exception e) {
                return theDate;
            }
            ResultSet dateRes = theDB.executeSelection(query);
            if (areDates) {
                try {
                    while (dateRes.next()) { // process results one row at a time

                        theDate = dateRes.getTimestamp(1);
                    }
                    dateRes.close();
                } catch (Exception e) {
                    return theDate;
                }
            }
            lastNewsDates = theDate;
            return theDate;
        } else {
            return lastNewsDates;
        }
    }

    public int InsertEntry(int feedID, String title, String description,
            String link, Date pubDate, String titleLemmas, String descLemmas) throws NoBaseException {
        String s = title+description+link;
        //System.out.println("Insertion d'une entrée lemmatisée : "+s);
        String t = s;
        s = Codecs.escapeHTML(s);
        s = Codecs.flushNormalText(s);
        if(!s.equals("")){
            Out.printErreur("Dans : "+t);
            String att = " Attention nouveau(x) caractère(s) à échapper : ";
            for(int j=0; j<s.length(); j++){
                att += "'"+s.charAt(j)+"' ["+new Integer(s.charAt(j)).toString()+"] ";
            }
            Out.printErreur(att);
        }
        lastFeedDates[feedID] = null;
        String query = String.format("INSERT INTO rss_articles (flux, datePubli, " +
                "titre, description, url, aEteLemmatise, aEteCapture, capture, titreLemma, descLemma) " +
                "VALUES (%d, '%tY-%tm-%td %tH:%tM:%tS' , '%s', '%s', '%s', 1, 0, '%s', '%s', '%s');",
                feedID, pubDate, pubDate, pubDate, pubDate, pubDate, pubDate,
                Codecs.escapeHTML(title),
                Codecs.escapeHTML(description),
                Codecs.echappeSQL(Codecs.desEscapeURL(link)),
                "",
                Codecs.escapeHTML(titleLemmas),
                Codecs.escapeHTML(descLemmas));
        //System.out.println("Entrée dans la base...");
        return theDB.executeInsertOrUpdate(query);
    }

    public int InsertEntry(int feedID, String title,
            String description, String link, Date pubDate) throws NoBaseException {
        String s = title+description+link;
        //System.out.println("Insertion d'une entrée : "+s);
        String t = s;
        s = Codecs.escapeHTML(s);
        s = Codecs.flushNormalText(s);
        if(!s.equals("")){
            String att = " Attention nouveau(x) caractère(s) à échapper : ";
            for(int j=0; j<s.length(); j++){
                att += "'"+s.charAt(j)+"' ["+new Integer(s.charAt(j)).toString()+"] ";
            }
            att += " dans "+t;
            Out.printErreur(att);
        }
        lastFeedDates[feedID] = null;
        String query = String.format("INSERT INTO rss_articles (flux, datePubli, " +
                "titre, description, url, aEteLemmatise, aEteCapture, capture, titreLemma, descLemma) " +
                "VALUES (%d, '%tY-%tm-%td %tH:%tM:%tS' , '%s', '%s', '%s', 0, 0, '%s', '%s', '%s');",
                feedID, pubDate, pubDate, pubDate, pubDate, pubDate, pubDate,
                Codecs.escapeHTML(title),
                Codecs.escapeHTML(description),
                Codecs.echappeSQL(Codecs.desEscapeURL(link)),
                "", "", "");
        //System.out.println("Entrée dans la base...");
        return theDB.executeInsertOrUpdate(query);
    }

    public synchronized int UpdateLemmas(RSSItem item) throws NoBaseException {
        String query = String.format("UPDATE rss_articles SET aEteLemmatise=1, titreLemma = '%s', descLemma = '%s' WHERE id = %d LIMIT 1;",
                Codecs.escapeHTML(item.getTitleLemmes()),
                Codecs.escapeHTML(item.getDescLemmes()),
                item.getId());
        return theDB.executeInsertOrUpdate(query);
    }

    public Set<RSSItem> getAllRssItem() throws NoBaseException {
        String query = String.format("SELECT * FROM rss_articles ORDER BY datePubli DESC");
        ResultSet rs = theDB.executeSelection(query);
        Set<RSSItem> items = new HashSet<RSSItem>();
        try {
            while (rs.next()) {
                RSSItem ri = new RSSItem();
                ri.setId(rs.getInt("id"));
                ri.setIdFluxRSS(rs.getInt("flux"));
                ri.setPubDate(rs.getTimestamp("datePubli"));
                ri.setTitle(Codecs.desEscapeHTML(rs.getString("titre")));
                ri.setDescription(Codecs.desEscapeHTML(rs.getString("description")));
                ri.setLink(Codecs.desEscapeHTML(rs.getString("url")));
                ri.setDescLemmes(Codecs.desEscapeHTML(rs.getString("descLemma")));
                ri.setTitleLemmes(Codecs.desEscapeHTML(rs.getString("titreLemma")));
                items.add(ri);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return items;
    }

    public Set<RSSItem> getAnAmountOfRssItem(int amount) throws NoBaseException {
        String query = String.format("SELECT * FROM rss_articles ORDER BY datePubli DESC LIMIT " + amount);
        ResultSet rs = theDB.executeSelection(query);
        Set<RSSItem> items = new HashSet<RSSItem>();
        try {
            while (rs.next()) {
                RSSItem ri = new RSSItem();
                ri.setId(rs.getInt("id"));
                ri.setIdFluxRSS(rs.getInt("flux"));
                ri.setPubDate(rs.getTimestamp("datePubli"));
                ri.setTitle(Codecs.desEscapeHTML(rs.getString("titre")));
                ri.setDescription(Codecs.desEscapeHTML(rs.getString("description")));
                ri.setLink(Codecs.desEscapeHTML(rs.getString("url")));
                ri.setDescLemmes(Codecs.desEscapeHTML(rs.getString("descLemma")));
                ri.setTitleLemmes(Codecs.desEscapeHTML(rs.getString("titreLemma")));
                items.add(ri);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return items;
    }

    public int[] getNbArticlesSurUnMois(int feedId) throws NoBaseException {
        int[] res = new int[30];
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(System.currentTimeMillis() - 30l * 24l * 60l * 60l * 1000l);
        String a = "" + cal.get(GregorianCalendar.YEAR);
        a = ((a.length() < 2) ? "000" : ((a.length() < 3) ? "00" : ((a.length() < 4) ? "0" : ""))) + a;
        String m = "" + (cal.get(GregorianCalendar.MONTH) + 1);
        m = ((m.length() < 2) ? "0" : "") + m;
        String j = "" + cal.get(GregorianCalendar.DAY_OF_MONTH);
        j = ((j.length() < 2) ? "0" : "") + j;
        cal.setTimeInMillis(System.currentTimeMillis());
        String a2 = "" + cal.get(GregorianCalendar.YEAR);
        a2 = ((a2.length() < 2) ? "000" : ((a2.length() < 3) ? "00" : ((a2.length() < 4) ? "0" : ""))) + a2;
        String m2 = "" + (cal.get(GregorianCalendar.MONTH) + 1);
        m2 = ((m2.length() < 2) ? "0" : "") + m2;
        String j2 = "" + cal.get(GregorianCalendar.DAY_OF_MONTH);
        j2 = ((j2.length() < 2) ? "0" : "") + j2;

        String query = String.format("SELECT DAYOFMONTH(datePubli), COUNT(id) " +
                "FROM rss_articles WHERE flux=" + feedId + " AND datePubli BETWEEN '" +
                a + "-" + m + "-" + j + "' AND '" +
                a2 + "-" + m2 + "-" + j2 + "' GROUP BY TO_DAYS(datePubli) ORDER BY TO_DAYS(datePubli) DESC;");
        ResultSet rs = theDB.executeSelection(query);
        // Jour de départ
        int depart = cal.get(GregorianCalendar.DAY_OF_MONTH);
        // Nombre de jours du mois précédent
        cal.setTimeInMillis(System.currentTimeMillis() - 30l * 24l * 60l * 60l * 1000l);
        int nbJours = (cal.getActualMaximum(GregorianCalendar.DAY_OF_MONTH));
        try {
            while (rs.next()) {
                int date = rs.getInt(1);
                int index = -(date-depart);
                if(index<0){
                    index = -(date-nbJours-depart);
                }
                if(index<30){
                    res[index] = rs.getInt(2);
                }
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("sql error : " + e.getMessage()+"\n"+e.getStackTrace());
        }
        return res;
    }

    public int[] getNbCollectesSurUnMois(int feedId) throws NoBaseException {
        int[] res = new int[30];
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(System.currentTimeMillis() - 30l * 24l * 60l * 60l * 1000l);
        String a = "" + cal.get(GregorianCalendar.YEAR);
        a = ((a.length() < 2) ? "000" : ((a.length() < 3) ? "00" : ((a.length() < 4) ? "0" : ""))) + a;
        String m = "" + (cal.get(GregorianCalendar.MONTH) + 1);
        m = ((m.length() < 2) ? "0" : "") + m;
        String j = "" + cal.get(GregorianCalendar.DAY_OF_MONTH);
        j = ((j.length() < 2) ? "0" : "") + j;
        cal.setTimeInMillis(System.currentTimeMillis());
        String a2 = "" + cal.get(GregorianCalendar.YEAR);
        a2 = ((a2.length() < 2) ? "000" : ((a2.length() < 3) ? "00" : ((a2.length() < 4) ? "0" : ""))) + a2;
        String m2 = "" + (cal.get(GregorianCalendar.MONTH) + 1);
        m2 = ((m2.length() < 2) ? "0" : "") + m2;
        String j2 = "" + cal.get(GregorianCalendar.DAY_OF_MONTH);
        j2 = ((j2.length() < 2) ? "0" : "") + j2;

        String query = String.format("SELECT DAYOFMONTH(date), COUNT(id) " +
                "FROM rss_log WHERE flux=" + feedId + " AND date BETWEEN '" +
                a + "-" + m + "-" + j + "' AND '" +
                a2 + "-" + m2 + "-" + j2 + "' GROUP BY TO_DAYS(date) ORDER BY TO_DAYS(date) DESC;");
        ResultSet rs = theDB.executeSelection(query);
        // Jour de départ
        int depart = cal.get(GregorianCalendar.DAY_OF_MONTH);
        // Nombre de jours du mois précédent
        cal.setTimeInMillis(System.currentTimeMillis() - 30l * 24l * 60l * 60l * 1000l);
        int nbJours = (cal.getActualMaximum(GregorianCalendar.DAY_OF_MONTH));
        try {
            while (rs.next()) {
                int date = rs.getInt(1);
                int index = -(date-depart);
                if(index<0){
                    index = -(date-nbJours-depart);
                }
                if(index<30){
                    res[index] = rs.getInt(2);
                }
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("sql error : " + e.getMessage()+"\n"+e.getStackTrace());
        }
        return res;
    }

    public int[] getNbCollectesSur24h(int feedId) throws NoBaseException {
        int[] res = new int[24];
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(System.currentTimeMillis() - 48l * 60l * 60l * 1000l);
        String a = "" + cal.get(GregorianCalendar.YEAR);
        a = ((a.length() < 2) ? "000" : ((a.length() < 3) ? "00" : ((a.length() < 4) ? "0" : ""))) + a;
        String m = "" + (cal.get(GregorianCalendar.MONTH) + 1);
        m = ((m.length() < 2) ? "0" : "") + m;
        String j = "" + cal.get(GregorianCalendar.DAY_OF_MONTH);
        j = ((j.length() < 2) ? "0" : "") + j;
        String h = "" + cal.get(GregorianCalendar.HOUR_OF_DAY);
        h = ((h.length() < 2) ? "0" : "") + h;
        cal.setTimeInMillis(System.currentTimeMillis());
        String a2 = "" + cal.get(GregorianCalendar.YEAR);
        a2 = ((a2.length() < 2) ? "000" : ((a2.length() < 3) ? "00" : ((a2.length() < 4) ? "0" : ""))) + a2;
        String m2 = "" + (cal.get(GregorianCalendar.MONTH) + 1);
        m2 = ((m2.length() < 2) ? "0" : "") + m2;
        String j2 = "" + cal.get(GregorianCalendar.DAY_OF_MONTH);
        j2 = ((j2.length() < 2) ? "0" : "") + j2;
        String h2 = "" + cal.get(GregorianCalendar.HOUR_OF_DAY);
        h2 = ((h2.length() < 2) ? "0" : "") + h2;

        String query = String.format("SELECT HOUR(date), COUNT(id) " +
                "FROM rss_log WHERE flux=" + feedId + " AND date BETWEEN '" +
                a + "-" + m + "-" + j + " "+h+":00:00' AND '" +
                a2 + "-" + m2 + "-" + j2 + " "+h2+":00:00' GROUP BY HOUR(date);");
        ResultSet rs = theDB.executeSelection(query);
        // Heure de départ
        int depart = cal.get(GregorianCalendar.HOUR);
        try {
            while (rs.next()) {
                int date = rs.getInt(1);
                int index = -(date-depart);
                if(index<0){
                    index = -(date-24-depart);
                }
                if(index<24){
                    res[index] = rs.getInt(2);
                }
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("sql error : " + e.getMessage()+"\n"+e.getStackTrace());
        }
        return res;
    }

    public Set<RSSItem> getAllRssItemsOfDate(int jj, int mm, int aaaa) throws NoBaseException {
        GregorianCalendar cal = new GregorianCalendar();
        cal.set(aaaa, mm, jj);
        //La date du lendemain
        cal.setTimeInMillis(cal.getTimeInMillis() + 86400000);
        String a = "" + cal.get(GregorianCalendar.YEAR);
        a = ((a.length() < 2) ? "000" : ((a.length() < 3) ? "00" : ((a.length() < 4) ? "0" : ""))) + a;
        String m = "" + cal.get(GregorianCalendar.MONTH);
        m = ((m.length() < 2) ? "0" : "") + m;
        String j = "" + cal.get(GregorianCalendar.DAY_OF_MONTH);
        j = ((j.length() < 2) ? "0" : "") + j;

        String query = String.format("SELECT * FROM rss_articles WHERE datePubli BETWEEN '" +
                aaaa + "-" + mm + "-" + jj + "' AND '" +
                a + "-" + m + "-" + j + "';");
        ResultSet rs = theDB.executeSelection(query);
        Set<RSSItem> items = new HashSet<RSSItem>();
        try {
            while (rs.next()) {
                RSSItem ri = new RSSItem();
                ri.setId(rs.getInt("id"));
                ri.setIdFluxRSS(rs.getInt("flux"));
                ri.setPubDate(rs.getTimestamp("datePubli"));
                ri.setTitle(Codecs.desEscapeHTML(rs.getString("titre")));
                ri.setDescription(Codecs.desEscapeHTML(rs.getString("description")));
                ri.setLink(Codecs.desEscapeHTML(rs.getString("url")));
                ri.setDescLemmes(Codecs.desEscapeHTML(rs.getString("descLemma")));
                ri.setTitleLemmes(Codecs.desEscapeHTML(rs.getString("titreLemma")));
                items.add(ri);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return items;
    }

    public Set<RSSItem> getAllRssItemsBetween(int jj1, int mm1, int aaaa1, int jj2, int mm2, int aaaa2) throws NoBaseException {
        GregorianCalendar cal = new GregorianCalendar();
        cal.set(aaaa2, mm2, jj2);
        //La date du lendemain de la fin
        cal.setTimeInMillis(cal.getTimeInMillis() + 86400000);
        String a = "" + cal.get(GregorianCalendar.YEAR);
        a = ((a.length() < 2) ? "000" : ((a.length() < 3) ? "00" : ((a.length() < 4) ? "0" : ""))) + a;
        String m = "" + cal.get(GregorianCalendar.MONTH);
        m = ((m.length() < 2) ? "0" : "") + m;
        String j = "" + cal.get(GregorianCalendar.DAY_OF_MONTH);
        j = ((j.length() < 2) ? "0" : "") + j;

        String query = String.format("SELECT * FROM rss_articles WHERE datePubli BETWEEN '" +
                aaaa1 + "-" + mm1 + "-" + jj1 + "' AND '" +
                a + "-" + m + "-" + j + "';");
        ResultSet rs = theDB.executeSelection(query);
        Set<RSSItem> items = new HashSet<RSSItem>();
        try {
            while (rs.next()) {
                RSSItem ri = new RSSItem();
                ri.setId(rs.getInt("id"));
                ri.setIdFluxRSS(rs.getInt("flux"));
                ri.setPubDate(rs.getTimestamp("datePubli"));
                ri.setTitle(Codecs.desEscapeHTML(rs.getString("titre")));
                ri.setDescription(Codecs.desEscapeHTML(rs.getString("description")));
                ri.setLink(Codecs.desEscapeHTML(rs.getString("url")));
                ri.setDescLemmes(Codecs.desEscapeHTML(rs.getString("descLemma")));
                ri.setTitleLemmes(Codecs.desEscapeHTML(rs.getString("titreLemma")));
                items.add(ri);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return items;
    }

    public int getNbSujetsEntre(Date d1, Date d2) throws NoBaseException {
        String query = String.format("SELECT COUNT(id) FROM rss_sujets " +
                "WHERE datePubli BETWEEN '%tY-%tm-%td' AND '%tY-%tm-%td';",
                d1, d1, d1, d2, d2, d2);
        ResultSet rs = theDB.executeSelection(query);
        int result = 0;
        try {
            while (rs.next()) {
                result = rs.getInt(1);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println(e.toString());
        }
        return result;
    }

    public SujetBrut[] getSujetsEntre(Date d1, Date d2) throws NoBaseException {
        Vector<SujetBrut> result = new Vector<SujetBrut>();
        String query = String.format("SELECT id, titre FROM rss_sujets " +
                "WHERE datePubli BETWEEN '%tY-%tm-%td' AND '%tY-%tm-%td';",
                d1, d1, d1, d2, d2, d2);
        ResultSet rs = theDB.executeSelection(query);
        try {
            while (rs.next()) {
                int id = rs.getInt("id");
                String titre = Codecs.desEscapeHTML(rs.getString("titre"));
                result.addElement(new SujetBrut(id, titre));
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println(e.toString());
        }
        for(int i=0; i<result.size(); i++){
            result.elementAt(i).setArticles(getArticlesDuSujet(result.elementAt(i).getId()));
        }
        
        // Fusion des sujets identiques
        int fusions = 1;
        while (fusions > 0) {
            fusions = 0;
            Vector<SujetBrut> result2 = new Vector<SujetBrut>();
            for (int i = 0; i < result.size(); i++) {
                boolean deja = false;
                for (int j = 0; j < result2.size(); j++) {
                    if (result.elementAt(i).equals(result2.elementAt(j))) {
                        deja = true;
                        result2.elementAt(j).fusionne(result.elementAt(i));
                        fusions++;
                        break;
                    }
                }
                if (!deja && result.elementAt(i).getArticles().length > 0) {
                    result2.addElement(result.elementAt(i));
                }
            }
            result = result2;
        }

        // Retour sous forme de tableau
        SujetBrut[] result0 = new SujetBrut[result.size()];
        result.toArray(result0);
        return result0;
    }

    private BaseRSSItem[] getArticlesDuSujet(int idSujet) throws NoBaseException {
        Vector<BaseRSSItem> result = new Vector<BaseRSSItem>();
        String query = "SELECT titre, description, url FROM rss_articlessujets " +
                "WHERE sujet="+ idSujet + ";";
        ResultSet rs = theDB.executeSelection(query);
        try {
            while (rs.next()) {
                BaseRSSItem item = new BaseRSSItem();
                item.setTitle(Codecs.desEscapeHTML(rs.getString("titre")));
                item.setDescription(Codecs.desEscapeHTML(rs.getString("description")));
                item.setLink(Codecs.desEscapeURL(rs.getString("url")));
                result.addElement(item);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println(e.toString());
        }
        BaseRSSItem[] result2 = new BaseRSSItem[result.size()];
        result.toArray(result2);
        return result2;
    }

    //private static Vector<RSSItem> tousLesArticles = null;
    public SujetTraite trouveArticlesDesSujets(SujetBrut sujets) throws NoBaseException {
        /*if(tousLesArticles == null){
            Out.printInfo("Chargement des articles en mémoire pour accélérer le processus.");
            tousLesArticles = new Vector<RSSItem>();
            String query = "SELECT id, flux, titre, description, url, datePubli " +
                    "FROM rss_articles;";
            ResultSet rs = theDB.executeSelection(query);
            try {
                while (rs.next()) {
                    RSSItem art = new RSSItem();
                    art.setId(rs.getInt("id"));
                    art.setIdFluxRSS(rs.getInt("flux"));
                    art.setTitle(Codecs.desEscapeHTML(rs.getString("titre")));
                    art.setDescription(Codecs.desEscapeHTML(rs.getString("description")));
                    art.setLink(Codecs.desEscapeHTML(rs.getString("url")));
                    art.setPubDate(rs.getDate("datePubli"));
                    tousLesArticles.addElement(art);
                }
                rs.close();
            } catch (SQLException e) {
                e.printStackTrace();
            }
            Out.printInfo("Chargement des articles en mémoire terminé ("+tousLesArticles.size()+" articles chargés).");
        }*/
        SujetTraite nouveau = new SujetTraite(sujets);
        Vector<BaseRSSItem> articles = new Vector<BaseRSSItem>();
        for (int j = 0; j < sujets.getArticles().length; j++) {
            String query = "SELECT id, flux, titre, description, url, datePubli " +
                    "FROM rss_articles " +
                    "WHERE url='" + Codecs.echappeSQL(Codecs.desEscapeURL(sujets.getArticles()[j].getLink())) + "';";
            ResultSet rs = theDB.executeSelection(query);
            try {
                while (rs.next()) {
                    RSSItem art = new RSSItem();
                    art.setId(rs.getInt("id"));
                    art.setIdFluxRSS(rs.getInt("flux"));
                    art.setTitle(Codecs.desEscapeHTML(rs.getString("titre")));
                    art.setDescription(Codecs.desEscapeHTML(rs.getString("description")));
                    art.setLink(Codecs.desEscapeHTML(rs.getString("url")));
                    art.setPubDate(rs.getTimestamp("datePubli"));
                    articles.addElement(art);
                }
                rs.close();
            } catch (SQLException e) {
                e.printStackTrace();
            }
            /*for(int k=0; k<tousLesArticles.size(); k++){
            if(tousLesArticles.elementAt(k).getLink().equalsIgnoreCase(
            Codecs.echappeSQL(Codecs.desEscapeURL(sujets.getUrls()[j])))){
            articles.addElement(tousLesArticles.elementAt(k));
            }
            }*/
        }
        RSSItem[] articles2 = new RSSItem[articles.size()];
        articles.toArray(articles2);
        for (int j = 0; j < articles2.length; j++) {
            articles2[j].setNomFluxRSS(getFeedName(articles2[j].getIdFluxRSS()));
        }
        nouveau.setArticles(articles2);
        return nouveau;
    }

    public int getRssItemCount() throws NoBaseException {
        String query = String.format("SELECT COUNT(id) FROM rss_articles");
        ResultSet rs = theDB.executeSelection(query);
        int result = 0;
        try {
            while (rs.next()) {
                result = rs.getInt(1);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return result;
    }

    public int getRssItemMaxID() throws NoBaseException {
        String query = String.format("SELECT MAX(id) FROM rss_articles");
        ResultSet rs = theDB.executeSelection(query);
        int result = 0;
        try {
            while (rs.next()) {
                result = rs.getInt(1);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return result;
    }

    public Vector<RSSItemCount> getRssItemCountByFeed() throws NoBaseException {
        System.out.println(" Flux avec publication");
        String query = String.format(
                "SELECT f.nom, COUNT(a.id), MAX(a.datePubli) " +
                "FROM rss_articles a, rss_flux f " +
                "WHERE f.id = a.flux " +
                "GROUP BY f.id " +
                "ORDER BY f.id ASC;");
        System.out.println("  Interrogation de la base");
        ResultSet rs = theDB.executeSelection(query);
        System.out.println("  Interrogation de la base terminée");
        Vector<RSSItemCount> items = new Vector<RSSItemCount>();
        try {
            while (rs.next()) {
                RSSItemCount ri = new RSSItemCount(RSSItemCount.DAY, rs.getString(1), rs.getString(3), rs.getInt(2));
                items.add(ri);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        System.out.println(" Flux sans publication");
        query = String.format(
                "SELECT f.nom " +
                "FROM rss_flux f " +
                "WHERE f.id NOT IN (" +
                "SELECT f.id " +
                "FROM rss_articles a, rss_flux f " +
                "WHERE f.id = a.flux);");
        System.out.println("  Interrogation de la base");
        rs = theDB.executeSelection(query);
        System.out.println("  Interrogation de la base terminée");
        try {
            while (rs.next()) {
                RSSItemCount ri = new RSSItemCount(RSSItemCount.DAY, rs.getString(1), "", 0);
                items.add(ri);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return items;
    }

    public Vector<RSSItemCount> getRssItemCountByDay() throws NoBaseException {
        String query = String.format(
                "SELECT datePubli, COUNT(id) " +
                "FROM rss_articles " +
                "GROUP BY DAYOFYEAR(datePubli) " +
                "ORDER BY datePubli DESC;");
        ResultSet rs = theDB.executeSelection(query);
        Vector<RSSItemCount> items = new Vector<RSSItemCount>();
        try {
            while (rs.next()) {
                RSSItemCount ri = new RSSItemCount(RSSItemCount.DAY, "", rs.getString(1), rs.getInt(2));
                items.add(ri);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return items;
    }

    public Vector<RSSItemCount> getRssItemCountByHour() throws NoBaseException {
        String query = String.format(
                "SELECT datePubli, COUNT(id) " +
                "FROM rss_articles " +
                "GROUP BY HOUR(datePubli);");
        ResultSet rs = theDB.executeSelection(query);
        Vector<RSSItemCount> items = new Vector<RSSItemCount>();
        try {
            while (rs.next()) {
                RSSItemCount ri = new RSSItemCount(RSSItemCount.HOUR, "", rs.getString(1), rs.getInt(2));
                items.add(ri);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return items;
    }

    public Set<RSSItem> getRssDate(String id) throws NoBaseException {
        String query = String.format("SELECT f.nom, i.datePubli, i.id " +
                "FROM rss_articles i, rss_flux f " +
                "WHERE i.flux = f.id " +
                "AND f.id = " + id + "");
        ResultSet rs = theDB.executeSelection(query);
        Set<RSSItem> items = new HashSet<RSSItem>();
        try {
            while (rs.next()) {
                RSSItem ri = new RSSItem();
                ri.setId(rs.getInt("id"));
                ri.setPubDate(rs.getTimestamp("datePubli"));
                ri.setTitle(Codecs.desEscapeHTML(rs.getString("nom")));
                items.add(ri);
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return items;
    }

    public RSSItem getRssItem(int id) throws NoBaseException {
        String query = String.format("SELECT titre, datePubli, id, description, titreLemma, descLemma " +
                "FROM rss_articles " +
                "WHERE id = " + id + "");
        ResultSet rs = theDB.executeSelection(query);
        RSSItem item = new RSSItem();
        try {
            while (rs.next()) {
                item.setId(rs.getInt("id"));
                item.setPubDate(rs.getTimestamp("datePubli"));
                item.setTitle(Codecs.desEscapeHTML(rs.getString("titre")));
                item.setDescription(Codecs.desEscapeHTML(rs.getString("description")));
                item.setTitleLemmes(Codecs.desEscapeHTML(rs.getString("titreLemma")));
                item.setDescLemmes(Codecs.desEscapeHTML(rs.getString("descLemma")));
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return item;
    }

    public RSSItem getEscapedRssItem(int id) throws NoBaseException {
        String query = String.format("SELECT titre, datePubli, id, description, titreLemma, descLemma " +
                "FROM rss_articles " +
                "WHERE id = " + id + ";");
        ResultSet rs = theDB.executeSelection(query);
        RSSItem item = new RSSItem();
        try {
            while (rs.next()) {
                item.setId(rs.getInt("id"));
                item.setPubDate(rs.getTimestamp("datePubli"));
                item.setTitle(rs.getString("titre"));
                item.setDescription(rs.getString("description"));
                item.setTitleLemmes(rs.getString("titreLemma"));
                item.setDescLemmes(rs.getString("descLemma"));
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
        }
        return item;
    }

    public Set<LemmaItem> getLemmas(int idArticle) throws NoBaseException {
        flush();
        Set<LemmaItem> lemmas = new LinkedHashSet<LemmaItem>();
        String query = String.format("SELECT * FROM rss_lemmes WHERE article = " + idArticle + ";");
        ResultSet lemmasRS = theDB.executeSelection(query);
        try {
            while (lemmasRS.next()) {
                LemmaItem li = new LemmaItem();
                li.setLemmaName(Codecs.desEscapeHTML(lemmasRS.getString("lemme")));
                li.setNodeID(lemmasRS.getInt("article"));
                li.setCountTitle(lemmasRS.getInt("occuTitre"));
                li.setCountDesc(lemmasRS.getInt("occuDesc"));
                li.setLemmaLex(Codecs.desEscapeHTML(lemmasRS.getString("nature")));
                lemmas.add(li);
            }
            lemmasRS.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        return lemmas;
    }

    public Set<LemmaItem> getNameLemmas(int idArticle) throws NoBaseException {
        flush();
        Set<LemmaItem> lemmas = new LinkedHashSet<LemmaItem>();
        String query = String.format("SELECT * FROM rss_lemmes WHERE article = " + idArticle + ";");
        ResultSet lemmasRS = theDB.executeSelection(query);
        try {
            while (lemmasRS.next()) {
                LemmaItem li = new LemmaItem();
                li.setLemmaName(Codecs.desEscapeHTML(lemmasRS.getString("lemme")));
                li.setNodeID(lemmasRS.getInt("article"));
                li.setCountTitle(lemmasRS.getInt("occuTitre"));
                li.setCountDesc(lemmasRS.getInt("occuDesc"));
                li.setLemmaLex(Codecs.desEscapeHTML(lemmasRS.getString("nature")));
                if(li.getLemmaLex().equalsIgnoreCase("NOM") ||
                   li.getLemmaLex().equalsIgnoreCase("NAM")) {
                    lemmas.add(li);
                }
            }
            lemmasRS.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        return lemmas;
    }

    public RSSItem[] getAllArticlesNonLemmatises() throws NoBaseException {
        String query = "SELECT DISTINCT id, titre, description FROM rss_articles " +
                "WHERE aEteLemmatise = 0 ORDER BY id DESC;";
        ResultSet aLemmatiser = theDB.executeSelection(query);
        Vector<BaseRSSItem> res = new Vector<BaseRSSItem>();
        try {
            while (aLemmatiser.next()) {
                RSSItem bri = new RSSItem();
                bri.setId(aLemmatiser.getInt("id"));
                bri.setTitle(Codecs.desEscapeHTML(aLemmatiser.getString("titre")));
                bri.setDescription(Codecs.desEscapeHTML(aLemmatiser.getString("description")));
                res.addElement(bri);
            }
            aLemmatiser.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        RSSItem[] res2 = new RSSItem[res.size()];
        res.toArray(res2);
        return res2;
    }

    public RSSItem[] getAllArticlesNonLemmatisesAPartirDe(Date date) throws NoBaseException {
        String query = String.format("SELECT DISTINCT id, titre, description FROM rss_articles " +
                "WHERE aEteLemmatise = 0 AND datePubli >= '%tY-%tm-%td' ORDER BY id ASC;",
                date, date, date);
        ResultSet aLemmatiser = theDB.executeSelection(query);
        Vector<BaseRSSItem> res = new Vector<BaseRSSItem>();
        try {
            while (aLemmatiser.next()) {
                RSSItem bri = new RSSItem();
                bri.setId(aLemmatiser.getInt("id"));
                bri.setTitle(Codecs.desEscapeHTML(aLemmatiser.getString("titre")));
                bri.setDescription(Codecs.desEscapeHTML(aLemmatiser.getString("description")));
                res.addElement(bri);
            }
            aLemmatiser.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        RSSItem[] res2 = new RSSItem[res.size()];
        res.toArray(res2);
        return res2;
    }

    public RSSItem[] getAllArticlesNonLemmatisesDu(Date date) throws NoBaseException {
        Date date2 = new Date();
        date2.setTime(date.getTime()+1000*60*60*24);
        String query = String.format("SELECT DISTINCT id, titre, description FROM rss_articles " +
                "WHERE aEteLemmatise = 0 AND datePubli BETWEEN '%tY-%tm-%td' AND '%tY-%tm-%td' ORDER BY id ASC;",
                date, date, date, date2, date2, date2);
        ResultSet aLemmatiser = theDB.executeSelection(query);
        Vector<BaseRSSItem> res = new Vector<BaseRSSItem>();
        try {
            while (aLemmatiser.next()) {
                RSSItem bri = new RSSItem();
                bri.setId(aLemmatiser.getInt("id"));
                bri.setTitle(Codecs.desEscapeHTML(aLemmatiser.getString("titre")));
                bri.setDescription(Codecs.desEscapeHTML(aLemmatiser.getString("description")));
                res.addElement(bri);
            }
            aLemmatiser.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        RSSItem[] res2 = new RSSItem[res.size()];
        res.toArray(res2);
        return res2;
    }

    public RSSItem[] getAllLiensSansCapture(SousCorpus tagsAcceptes) throws NoBaseException {
        Vector<Tag> listeCategories = new Vector<Tag>();
        Vector<Tag> listePays = new Vector<Tag>();
        Vector<Tag> listeCouvertures = new Vector<Tag>();
        Vector<Tag> listeInformations = new Vector<Tag>();

        for(int i=0; i<tagsAcceptes.listeTagsAcceptes().length; i++){
            if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_CATEGORIE && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_PAYS){
                listeCategories.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_PAYS && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_COUVERTURE){
                listePays.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_COUVERTURE && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_INFORMATION){
                listeCouvertures.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_INFORMATION && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_EXTERNE){
                listeInformations.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            }
        }

        String s = "SELECT DISTINCT url, id FROM rss_articles " +
                "WHERE aEteCapture = 0 AND flux IN (" +
                "SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND ";
        s += "(t.tag=";
        for(int i=0; i<listeCategories.size(); i++){
            s += ""+listeCategories.elementAt(i).getId();
            if(i < listeCategories.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listePays.size(); i++){
            s += ""+listePays.elementAt(i).getId();
            if(i < listePays.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeCouvertures.size(); i++){
            s += ""+listeCouvertures.elementAt(i).getId();
            if(i < listeCouvertures.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeInformations.size(); i++){
            s += ""+listeInformations.elementAt(i).getId();
            if(i < listeInformations.size()-1){
                s += " OR t.tag=";
            }
        }
        s += "))))) ORDER BY id DESC;";
        /*String query = "SELECT DISTINCT url, id FROM rss_articles " +
                "WHERE aEteCapture = 0 ORDER BY id DESC;";*/
        ResultSet liens = theDB.executeSelection(s);
        Vector<RSSItem> res = new Vector<RSSItem>();
        try {
            while (liens.next()) {
                RSSItem item = new RSSItem();
                item.setLink(liens.getString("url").trim());
                item.setId(liens.getInt("id"));
                res.addElement(item);
            }
            liens.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        RSSItem[] res2 = new RSSItem[res.size()];
        res.toArray(res2);
        return res2;
    }

    public RSSItem[] getAllLiensSansCaptureAPartirDe(Date date, SousCorpus tagsAcceptes) throws NoBaseException {
        Vector<Tag> listeCategories = new Vector<Tag>();
        Vector<Tag> listePays = new Vector<Tag>();
        Vector<Tag> listeCouvertures = new Vector<Tag>();
        Vector<Tag> listeInformations = new Vector<Tag>();

        for(int i=0; i<tagsAcceptes.listeTagsAcceptes().length; i++){
            if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_CATEGORIE && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_PAYS){
                listeCategories.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_PAYS && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_COUVERTURE){
                listePays.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_COUVERTURE && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_INFORMATION){
                listeCouvertures.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_INFORMATION && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_EXTERNE){
                listeInformations.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            }
        }

        String s = String.format("SELECT DISTINCT url, id FROM rss_articles " +
                "WHERE aEteCapture = 0 AND datePubli >= '%tY-%tm-%td' AND flux IN (" +
                "SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND ",
                date, date, date);
        s += "(t.tag=";
        for(int i=0; i<listeCategories.size(); i++){
            s += ""+listeCategories.elementAt(i).getId();
            if(i < listeCategories.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listePays.size(); i++){
            s += ""+listePays.elementAt(i).getId();
            if(i < listePays.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeCouvertures.size(); i++){
            s += ""+listeCouvertures.elementAt(i).getId();
            if(i < listeCouvertures.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeInformations.size(); i++){
            s += ""+listeInformations.elementAt(i).getId();
            if(i < listeInformations.size()-1){
                s += " OR t.tag=";
            }
        }
        s += "))))) ORDER BY id ASC;";
        /*String query = String.format("SELECT DISTINCT url, id FROM rss_articles " +
                "WHERE aEteCapture = 0 AND datePubli >= '%tY-%tm-%td' ORDER BY id ASC;",
                date, date, date);*/
        ResultSet liens = theDB.executeSelection(s);
        Vector<RSSItem> res = new Vector<RSSItem>();
        try {
            while (liens.next()) {
                RSSItem item = new RSSItem();
                item.setLink(liens.getString("url").trim());
                item.setId(liens.getInt("id"));
                res.addElement(item);
            }
            liens.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        RSSItem[] res2 = new RSSItem[res.size()];
        res.toArray(res2);
        return res2;
    }

    public RSSItem[] getAllLiensSansCaptureDu(Date date, SousCorpus tagsAcceptes) throws NoBaseException {
        Date date2 = new Date();
        date2.setTime(date.getTime()+1000*60*60*24);
        Vector<Tag> listeCategories = new Vector<Tag>();
        Vector<Tag> listePays = new Vector<Tag>();
        Vector<Tag> listeCouvertures = new Vector<Tag>();
        Vector<Tag> listeInformations = new Vector<Tag>();

        for(int i=0; i<tagsAcceptes.listeTagsAcceptes().length; i++){
            if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_CATEGORIE && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_PAYS){
                listeCategories.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_PAYS && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_COUVERTURE){
                listePays.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_COUVERTURE && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_INFORMATION){
                listeCouvertures.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_INFORMATION && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_EXTERNE){
                listeInformations.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            }
        }

        String s = String.format("SELECT DISTINCT url, id FROM rss_articles " +
                "WHERE aEteCapture = 0 AND datePubli BETWEEN '%tY-%tm-%td' AND '%tY-%tm-%td' " +
                "AND flux IN (" +
                "SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND ",
                date, date, date, date2, date2, date2);
        s += "(t.tag=";
        for(int i=0; i<listeCategories.size(); i++){
            s += ""+listeCategories.elementAt(i).getId();
            if(i < listeCategories.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listePays.size(); i++){
            s += ""+listePays.elementAt(i).getId();
            if(i < listePays.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeCouvertures.size(); i++){
            s += ""+listeCouvertures.elementAt(i).getId();
            if(i < listeCouvertures.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeInformations.size(); i++){
            s += ""+listeInformations.elementAt(i).getId();
            if(i < listeInformations.size()-1){
                s += " OR t.tag=";
            }
        }
        s += "))))) ORDER BY id ASC;";
        /*String query = String.format("SELECT DISTINCT url, id FROM rss_articles " +
                "WHERE aEteCapture = 0 AND datePubli BETWEEN '%tY-%tm-%td' AND '%tY-%tm-%td' ORDER BY id ASC;",
                date, date, date, date2, date2, date2);*/
        ResultSet liens = theDB.executeSelection(s);
        Vector<RSSItem> res = new Vector<RSSItem>();
        try {
            while (liens.next()) {
                RSSItem item = new RSSItem();
                item.setLink(liens.getString("url").trim());
                item.setId(liens.getInt("id"));
                res.addElement(item);
            }
            liens.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        RSSItem[] res2 = new RSSItem[res.size()];
        res.toArray(res2);
        return res2;
    }

    public void saveImageOfAllArticlesDe(Date debut, Date fin, SousCorpus tagsAcceptes) throws NoBaseException {
        String query = "TRUNCATE TABLE image_rss_articles;";
        theDB.executeDeletion(query);

        Vector<Tag> listeCategories = new Vector<Tag>();
        Vector<Tag> listePays = new Vector<Tag>();
        Vector<Tag> listeCouvertures = new Vector<Tag>();
        Vector<Tag> listeInformations = new Vector<Tag>();

        for(int i=0; i<tagsAcceptes.listeTagsAcceptes().length; i++){
            if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_CATEGORIE && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_PAYS){
                listeCategories.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_PAYS && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_COUVERTURE){
                listePays.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_COUVERTURE && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_INFORMATION){
                listeCouvertures.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            } else if(tagsAcceptes.listeTagsAcceptes()[i]>=TAGS_INFORMATION && tagsAcceptes.listeTagsAcceptes()[i]<TAGS_EXTERNE){
                listeInformations.addElement(new Tag(tagsAcceptes.listeTagsAcceptes()[i], ""));
            }
        }

        String s = String.format("SELECT a.id, f.nom, a.datePubli, a.titre, a.description, a.url " +
                "FROM rss_articles a, rss_flux f " +
                "WHERE a.flux=f.id AND a.datePubli BETWEEN '%tY-%tm-%td 00:00:00' AND '%tY-%tm-%td 00:00:00' " +
                "AND a.flux IN (" +
                "SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND ",
                debut, debut, debut, fin, fin, fin);
        s += "(t.tag=";
        for(int i=0; i<listeCategories.size(); i++){
            s += ""+listeCategories.elementAt(i).getId();
            if(i < listeCategories.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listePays.size(); i++){
            s += ""+listePays.elementAt(i).getId();
            if(i < listePays.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeCouvertures.size(); i++){
            s += ""+listeCouvertures.elementAt(i).getId();
            if(i < listeCouvertures.size()-1){
                s += " OR t.tag=";
            }
        }
        s += ") AND f.id IN (SELECT id FROM rss_flux f, rss_tagflux t WHERE t.flux=f.id AND (t.tag=";
        for(int i=0; i<listeInformations.size(); i++){
            s += ""+listeInformations.elementAt(i).getId();
            if(i < listeInformations.size()-1){
                s += " OR t.tag=";
            }
        }
        s += "))))) ORDER BY id ASC;";
        ResultSet liens = theDB.executeSelection(s);
        Vector<RSSItem> res = new Vector<RSSItem>();
        try {
            while (liens.next()) {
                RSSItem item = new RSSItem();
                item.setId(liens.getInt(1));
                item.setNomFluxRSS(Codecs.desEscapeHTML(liens.getString(2)));
                item.setPubDate(liens.getTimestamp(3));
                item.setTitle(Codecs.desEscapeHTML(liens.getString(4)));
                item.setDescription(Codecs.desEscapeHTML(liens.getString(5)));
                item.setLink(Codecs.desEscapeURL(liens.getString(6)));
                res.addElement(item);
            }
            liens.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        RSSItem[] res2 = new RSSItem[res.size()];
        res.toArray(res2);
        for(int i=0; i<res2.length; i++){
            String query2 = "INSERT INTO image_rss_articles ("+
                "id, nomFlux, datePubli, titre, description, url) VALUES (" +
                res2[i].getId()+", '"+
                Codecs.echappeSQL(res2[i].getNomFluxRSS())+"', "+
                String.format("'%tY-%tm-%td %tH:%tM:%tS'", res2[i].getPubDate(), res2[i].getPubDate(),
                res2[i].getPubDate(), res2[i].getPubDate(), res2[i].getPubDate(), res2[i].getPubDate())+", '"+
                Codecs.echappeSQL(res2[i].getTitle())+"', '"+
                Codecs.echappeSQL(res2[i].getDescription())+"', '"+
                Codecs.echappeSQL(res2[i].getLink())+"');";
            theDB.executeInsertOrUpdate(query2);
        }
    }

    public void captureLien(int id, String lien) {
        String captFile = "" + id;
        File f = data.base.Config.getCaptureProgram();
        final Thread t = Thread.currentThread();
        final Thread t2 = new Thread(){
            @Override
            public void run(){
                try{
                    Thread.sleep(40000);
                    t.interrupt();
                } catch(InterruptedException ie){

                }
            }
        };
        try{
            Process p = Runtime.getRuntime().exec(f.getAbsolutePath() + " --url=\"" +
                    lien + "\" --out=\"" + 
                    data.base.Config.getCaptureDirectory() + "\\" + captFile +
                    ".png\" --silent --max-wait=30000");
            Out.printInfo("Capture n°"+captFile+" lancée : "+lien);
            long time = System.currentTimeMillis();
            int exitValue = 0;
            // Il faut s'assurer que l'on n'est plus interrompu.
            Thread.currentThread().interrupted();
            try{
                t2.start();
                exitValue = p.waitFor();
                t2.interrupt();
            } catch(InterruptedException ie){
                t2.interrupt();
                p.destroy();
                throw new Exception("capture trop longue ("+((System.currentTimeMillis()-time)/1000)+" secondes).");
            }
            if(exitValue != 0){
                throw new Exception(f.getAbsolutePath()+", "+"erreur "+exitValue);
            }
            if(new File(data.base.Config.getCaptureDirectory() + "\\" + captFile +
                    ".png").exists()){
                String query = "UPDATE rss_articles SET aEteCapture=1, " +
                        "capture='"+captFile+"' " + "WHERE id="+id+";";
                theDB.executeInsertOrUpdate(query);
            } else {
                throw new Exception("pas d'image générée.");
            }
        } catch(IOException ioe){
            Out.printErreur("Capture n°"+captFile+" impossible : "+ioe.getMessage());
        } catch(Exception e){
            Out.printErreur("Le programme utilisé pour la capture n°"+captFile+" a échoué : "+e.getMessage());
        }
    }

    public int getLastInsertEntry() throws SQLException, NoBaseException  {
        String query = String.format("SELECT MAX(id) FROM rss_articles");
        ResultSet rs = theDB.executeSelection(query);
        int res = 0;
        try {
            rs.next();
            res = rs.getInt(1);
            rs.close();
            return res;
        } catch (SQLException e) {
            System.out.println("error on get idRSSitem last key " + e.toString());
            throw e;
        }
    }

    public synchronized int InsertLemma(int ifeedid, String lemma, int nbTitle, int nbDesc, String lex) throws NoBaseException {
        flushConditionnel();
        if (!unwantedLemma(lemma)) {
            String query = String.format("INSERT INTO temp_rss_lemmes(article, lemme, occuTitre, occuDesc, nature)  " +
                    "VALUES (%d,'%s',%d,%d, '%s')",
                    ifeedid, Codecs.escapeHTML(lemma), nbTitle, nbDesc, lex);
            return theDB.executeInsertOrUpdate(query);
        } else {
            return 0;
        }
    }

    public synchronized int logRead(int feedID, long date, int newItems, String error) throws NoBaseException {
        flushConditionnel();
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(date);
        int annee = cal.get(GregorianCalendar.YEAR);
        int mois = cal.get(GregorianCalendar.MONTH)+1;
        int jour = cal.get(GregorianCalendar.DAY_OF_MONTH);
        int heure = cal.get(GregorianCalendar.HOUR_OF_DAY);
        int minute = cal.get(GregorianCalendar.MINUTE);
        int seconde = cal.get(GregorianCalendar.SECOND);
        String dateTime = annee+"-"+mois+"-"+jour+" "+heure+":"+minute+":"+seconde;
        String query = "INSERT INTO temp_rss_log(flux, date, nouveaux, erreur) VALUES("+
                feedID+", '"+dateTime+"', "+newItems+", '"+Codecs.echappeSQL(error)+"');";
        return theDB.executeInsertOrUpdate(query);
    }

    private void flush() throws NoBaseException {
        // Vidange des tables temporaires
        String query = String.format("INSERT INTO rss_log SELECT * FROM temp_rss_log;");
        theDB.executeInsertOrUpdate(query);
        query = String.format("TRUNCATE temp_rss_log;");
        theDB.executeInsertOrUpdate(query);
        query = String.format("INSERT INTO rss_lemmes SELECT * FROM temp_rss_lemmes;");
        theDB.executeInsertOrUpdate(query);
        query = String.format("TRUNCATE temp_rss_lemmes;");
        theDB.executeInsertOrUpdate(query);
    }

    private void flushConditionnel() throws NoBaseException {
        String query = "SELECT COUNT(lemme) FROM temp_rss_lemmes;";
        ResultSet rs = theDB.executeSelection(query);
        int nb = 0;
        try {
            rs.next();
            nb = rs.getInt(1);
            rs.close();
        } catch (SQLException e) {
            e.printStackTrace();
        }
        if(nb>=5000){
            query = String.format("INSERT INTO rss_lemmes SELECT * FROM temp_rss_lemmes;");
            theDB.executeInsertOrUpdate(query);
            query = String.format("TRUNCATE temp_rss_lemmes;");
            theDB.executeInsertOrUpdate(query);
        }
        query = "SELECT COUNT(date) FROM temp_rss_log;";
        rs = theDB.executeSelection(query);
        nb = 0;
        try {
            rs.next();
            nb = rs.getInt(1);
            rs.close();
        } catch (SQLException e) {
            e.printStackTrace();
        }
        if(nb>=5000){
            query = String.format("INSERT INTO rss_log SELECT * FROM temp_rss_log;");
            theDB.executeInsertOrUpdate(query);
            query = String.format("TRUNCATE temp_rss_log;");
            theDB.executeInsertOrUpdate(query);
        }
    }

    public void nettoieURLs() throws NoBaseException {
        // URLs de Google
        String query = "SELECT DISTINCT url FROM rss_articlessujets;";
        ResultSet rs = theDB.executeSelection(query);
        Vector<String> listeURLs = new Vector<String>();
        try {
            while (rs.next()) {
                listeURLs.addElement(rs.getString(1).trim());
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        for(int i=0; i<listeURLs.size(); i++){
            Out.printInfo("Nettoyage Google : "+(i+1)+"/"+listeURLs.size());
            String s = listeURLs.elementAt(i);
            String t = listeURLs.elementAt(i);
            t = Codecs.echappeSQL(Codecs.desEscapeURL(t));
            String query2 = "UPDATE rss_articlessujets " +
                    "SET url='"+t+"' "+
                    "WHERE url='"+s+"';";
            theDB.executeInsertOrUpdate(query2);
        }
        // URLs de la base
        /*String*/ query = "SELECT DISTINCT url FROM rss_articles;";
        /*ResultSet*/ rs = theDB.executeSelection(query);
        /*Vector<String>*/ listeURLs = new Vector<String>();
        try {
            while (rs.next()) {
                listeURLs.addElement(rs.getString(1).trim());
            }
            rs.close();
        } catch (SQLException e) {
            System.out.println("error " + e.toString());
        }
        for(int i=0; i<listeURLs.size(); i++){
            Out.printInfo("Nettoyage Base : "+(i+1)+"/"+listeURLs.size());
            String s = listeURLs.elementAt(i);
            String t = listeURLs.elementAt(i);
            t = Codecs.echappeSQL(Codecs.desEscapeURL(t));
            String query2 = "UPDATE rss_articles " +
                    "SET url='"+t+"' "+
                    "WHERE url='"+s+"';";
            theDB.executeInsertOrUpdate(query2);
        }
    }

    private boolean unwantedLemma(String lemma) {
        //On n'utilise plus cette méthode qui élimine des lemmes utiles.
        //De toute façon, les lemmes sont éliminés par leur nature.
        return false;

        /*

        // aucun lemme de taille 1 n'est souhaitable (de toute façon, a/à/y sont éliminés ensuite)
        if(lemma.length()<2){
            return true;
        }
        // detection par le corpus de termes de la table ipri_lemmes_ignore
        if (ignoreLemma.contains(lemma)) {
            //System.out.println("lemme refusé: "+lemma);
            return true;
        }
        // elimination des valeurs numeriques extravagantes
        List weirdLemma = new ArrayList();
        weirdLemma.add("$");
        weirdLemma.add("&");
        weirdLemma.add("#");
        weirdLemma.add("+");
        weirdLemma.add("/");
        weirdLemma.add("@");
        weirdLemma.add("(");
        weirdLemma.add(")");
        weirdLemma.add("*");
        for (int i = 0; i < weirdLemma.size(); i++) {
            if (lemma.contains((CharSequence) weirdLemma.get(i))) {
                //System.out.println("lemme refusé : "+lemma);
                return true;
            }
        }
        return false;
        
        */
    }

    /*private void getIgnoreLemma() throws SQLException {
        if (ignoreLemma == null) {
            String query = String.format("SELECT lemme FROM ipri_lemmes_ignore");
            ResultSet rs = theDB.executeSelection(query);
            try {
                while (rs.next()) {
                    ignoreLemma.add(rs.getString(1));
                }
                rs.close();
            } catch (SQLException e) {
                System.out.println("error retrieving lemmas to ignore");
                throw e;
            }
        }
    }*/

    /*public synchronized int removeAllLemme() {
        String query = String.format("TRUNCATE ipri_lemmes");
        try {
            return theDB.executeInsertOrUpdate(query);
        } catch (Exception e) {
            System.out.println("error removeAllLemme " + e.toString());
            return 0;
        }

    }*/
}
